#include "llvm/Pass.h"
#include "llvm/IR/Function.h"
#include "llvm/Support/raw_ostream.h"
#include "llvm/IR/Instruction.h"
#include "llvm/IR/Instructions.h"
#include "llvm/IR/Operator.h"
#include "llvm/IR/IRBuilder.h"
#include "ConstantPropagationInstruction.h"

using namespace llvm;

namespace {
struct OurConstantPropagationPass : public FunctionPass {
  std::vector<Value *> Variables;
  std::vector<ConstantPropagationInstruction *> Instructions;

  static char ID;
  OurConstantPropagationPass() : FunctionPass(ID) {}

  void createConstantPropagationInstructions(Function &F) {
    for (BasicBlock &BB : F) {
      for (Instruction &I : BB) {
        ConstantPropagationInstruction *Instr = new ConstantPropagationInstruction(&I);
        Instructions.push_back(Instr);

        Instruction *Predecessor = I.getPrevNonDebugInstruction();
        if (Predecessor == nullptr) {
          for (BasicBlock *BBPredecessor : predecessors(&BB)) {
            Instr->addPredecessor(*std::find_if(Instructions.begin(), Instructions.end(),
            [BBPredecessor](auto &CPI){
                                return CPI->getInstruction() == BBPredecessor->getTerminator(); }));
          }
        }
        else {
          Instr->addPredecessor(Instructions[Instructions.size() - 2]);
        }
      }
    }
  }

  void findVariables(Function &F) {
    for (BasicBlock &BB : F) {
      for (Instruction &I : BB) {
        if (isa<AllocaInst>(&I)) {
          Variables.push_back(&I);
        }
      }
    }
  }

  void setVariables() {
    for (ConstantPropagationInstruction *CPI : Instructions) {
      CPI->addVariables(Variables);
    }
  }

  bool runOnFunction(Function &F) override {
    createConstantPropagationInstructions(F);
    findVariables(F);
    setVariables();

    return true;
  }
};
}

char OurConstantPropagationPass::ID = 0;
static RegisterPass<OurConstantPropagationPass> X("propagation", "Our simple constant propagation pass",
                             false /* Only looks at CFG */,
                             false /* Analysis Pass */);