#include "DinamickiNiz.h"

/* BITNO:
 * U programskom jeziku C++ postoji koncept koji se naziva RAII - Resource acquisition
 * is initialization. RAII je kljucni koncept za pravilno koriscenje dinamickih resursa.
 * 
 * RAII je koncept koji vezuje opseg zivota resursa koji se mora obezbediti pre upotrebe
 * za opseg zivota objekta. Takvi resursi su: alocirana memorija na hipu, niti, soketi,
 * fajlovi, muteksi i slicno.
 * 
 * RAII kao koncept garantuje da ce resurs biti dostupan bilo kojoj funkciji koja moze
 * da pristupi objektu, cime se izbegava proveravanje da li resurs postoji tokom izvrsavanja.
 * Preciznije, dostupnost resursa je invarijanta klase. Pored ovoga, RAII garantuje da se
 * svi resursi oslobadjaju kada se zavrsi opseg zivota objekta koji njima upravlja i to
 * suprotnim redosledom od njihove akvizicije. Posledica ovog koncepta je da se svi 
 * neophodni resursi moraju obezbediti u konstruktoru klase. 
 * 
 * U slucaju da akvizicija resursa ne uspe, tj. ako konstruktor baci izuzetak, tada ce svi
 * resursi koje koriste svi prethodno u potpunosti konstruisani atributi ili bazni objekti
 * biti potpuno oslobodjeni u redosledu suprotnom od akvizicije. 
 * 
 * Ukratko, RAII se moze ostvariti na sledeci nacin:
 * 1. Enkapsulirati svaki resurs u svoju klasu, pri cemu
 *		1.1 Konstruktor obezdedjuje resurs i uspostavlja neophodne invarijante klase ili
 *			baca izuzetak ako to ne moze da uradi.
 *		1.2 Destruktor oslobadja resurse i nikada ne baca izuzetke.
 * 2. Uvek treba da koristimo resurse pomocu neke instance RAII-klase koja ima
 *		2.1 Automatsko trajanje skladistenja ili privremeni opseg zivota.
 *	  ili
 *		2.2 Imam opseg zivota koji je ogranicen opsegom automatskog ili privremenog objekta.
 * 
 * Kako znati da klasa nije u skladu sa RAII?
 * 
 * Sve klase koje imaju kao metode funkcije oblika:
 * 1. open()/close() kojima se upravlja resursima mimo opsega zivota objekta.
 * 2. lock()/unlock() kojima se vrsi sinhronizacija mimo opsega zivota objekta.
 * 3. init()/destroy() kojima se kreira i unistava resurs mimo opsega zivota objekta.
 * 
 * Svi navedeni pristupi nisu RAII, jer ne vezuju dostupnost resursa za opseg zivota 
 * objekta, vec omogucavaju nezavisno upravljanje resursom. Prebacivanje upravljanja 
 * resursima na korisnika je lose, jer se oslanja na to da je korisnik odgovoran i da 
 * svojim ponasanjem nece izazvati curenje resursa. Pretpostavka odgovornosti korisnika 
 * je cesto uzrok katastrofalnih bagova u programima. 
 * 
 * Postovanje RAII koncepta pri dizajniranju klasa garantuje pravilno upravljanje 
 * resursima na nacin koji ne zavisi od odgovornosti ili vestine korsinika. Resurs 
 * se mora obezbediti u konstruktoru (ili se izbacuje izuzetak) i mora se osloboditi u
 * destruktoru na kraju zivota objekta. Na taj nacin se obezbedjuje da se prilikom
 * upotrebe objekta ne mora raditi eksplicitno oslobadjanje resursa, jer ce objekat 
 * to samostalno uciniti. Kao posledicu pravilne upotrebe RAII koncepta dobijamo to
 * da RAII saglasne tipove podataka mozemo koristiti na isti nacin kao sto bismo koistili
 * i primitivne tipove podataka (int, double itd), jer se upravljanje resursima desava
 * automatski.
 * 
 */

/* podrazumevani konstruktor */
DinamickiNiz::DinamickiNiz() {

	/* niz je na startu prazan */
	_size = 0;
	/* postavljamo podrazumevani kapacitet */
	_capacity = DEFAULT_SIZE;
	/* alociramo niz na hipu */
	_niz = new double[_capacity]; /* RAII - obezbedjivanje resursa se radi u konstruktoru */

	/* stampamo poruku da bismo mogli da pratimo opseg zivota objekta */
	std::cout << "Dinamicki niz: Kreiram objekat" << std::endl;
}

/* parametrizovani konstruktor koji kreira niz zadate duzine 
 * drugi argument je opcioni i ako se izostavi podrazumevano ima vrednost 0 
 * sto je definisano u deklaraciji klase
 */
DinamickiNiz::DinamickiNiz(int n, double val) {

	/* duzinu i kapacitet niza postavljamo na trazeni */
	_size = n;
	_capacity = n;
	/* alociramo niz na hipu */
	_niz = new double[_capacity]; /* RAII - obezbedjivanje resursa se radi u konstruktoru */

	/* inicijalizujemo niz */
	for (int i = 0; i < _size; i++)
		_niz[i] = val;
	
	/* stampamo poruku da bismo mogli da pratimo opseg zivota objekta */
	std::cout << "Dinamicki niz: Kreiram objekat" << std::endl;
}

/* Konstruktor kopije
 * Primetimo da argument prosledjujemo kao referencu da bismo izbegli nepotrebno
 * kopiranje.
 */
DinamickiNiz::DinamickiNiz(const DinamickiNiz& dn) {

	/* velicinu i kapacitet postavljamo na vrednosti iz niza dn */
	_size = dn._size;
	_capacity = dn._capacity;

	/* BITNO:
	 * Prilikom pravljanje kopija objekata razlikujemo dva tipa kopiranja:
	 * 1. Plitko kopiranje - kopiranje u kojem samo kopiramo reference/pokazivace 
	 *						 dinamickih resursa koje nasa klasa koristi. Posledica
	 *						 plitkog kopiranja u ovom slucaju bi bila to sto bi i objekat
	 *					     dn i ovaj novi objekat pokazivali na isti niz u memoriji, sto
	 *					     znaci da bi promena tog niza bila vidljiva i jednom i drugom
	 *					     objektu. Naredba kojom bi se izvrsilo plitko kopiranje je:
	 *						_niz = dn._niz;
	 * 
	 * 2. Duboko kopiranje - kopiranje u kojem se pravi potpuno nova kopira svakog 
	 *						 dinamickog resursa koji nasa klasa sadrzi. U ovom slucaju to
	 *					     bi znacilo da treba da kreiramo novi niz na hipu i zatim da
	 *						 prekopiramo sadrzaj niza dn u taj novi niz koji smo kreirali.
	 *					     Posledica takvog kopiranja je da dobijamo dva u potpunosti 
	 *						 nezavisna objekta i da izmena jednog niza nema nikakav uticaj
	 *						 na drugi. Upravo je duboko kopiranje ono sto nam treba u ovom
	 *						 slucaju. 
	 */

	/* duboko kopiranje - alociramo novi niz na hipu */
	_niz = new double[_capacity]; /* RAII - obezbedjivanje resursa se radi u konstruktoru */
	/* kopiramo svaki pojedinacni element */
	for (int i = 0; i < _size; i++)
		_niz[i] = dn._niz[i];

	/* stampamo poruku da bismo mogli da pratimo opseg zivota objekta */
	std::cout << "Dinamicki niz: Kreiram kopiju" << std::endl;
}

/* operator dodele - copy assignment */
DinamickiNiz& DinamickiNiz::operator =(const DinamickiNiz& dn) {

	/* stampamo poruku da bismo mogli da pratimo opseg zivota objekta */
	std::cout << "Dinamicki niz: Operator dodele" << std::endl;

	/* BITNO:
	 * Provera dodele samom sebi je kljucna sa klasama koje koriste dinamcke
	 * resurse. Ako bismo oslobodili resurs koji koristimo i zatim ponovo
	 * alocirali, dobili bismo nekonzistentno stanje objekta. 
	 * 
	 * Semantika operatora dodele nalaze da dodelu mozemo da izvrsimo samo ako
	 * se radi o razlicitim objekta. Dodeljivanje samom sebi je besmisleno. 
	 */
	if (this == &dn)
		return *this;

	/* ako je niz vec alociran */
	if (_niz != nullptr)
		/* oslobadjamo ga */
		delete[] _niz;

	/* velicinu i kapacitet postavljamo na vrednosti iz niza dn */
	_size = dn._size;
	_capacity = dn._capacity;

	/* alociramo novi niz na hipu */
	_niz = new double[_capacity];
	/* kopiramo sadrzaj niza dn */
	for (int i = 0; i < _size; i++)
		_niz[i] = dn._niz[i];

	/* kao rezultat vracamo referencu na tekuci objekat */
	return *this;

	/* BITNO:
	 * Primetimo da operator dodele ne narusava RAII koncept.
	 * Resurs je dostupan pre poziva, tokom i nakon poziva operatora dodele,
	 * pa invarijanta klase nije narusena. 
	 */
}

/* Destruktor:
 * Nasa klasa koristi dinamicki resurs, pa smo duzni da obezbedimo
 * eksplicitni destruktor u kojem cemo taj resurs osloboditi.
 */
DinamickiNiz::~DinamickiNiz() {

	/* RAII - oslobadjanje resursa se radi u destruktoru */
	delete[] _niz;

	/* BITNO:
	 * Cesta greska u C++ je pogresno oslobadjanje nizova naredbom
	 * 
	 * delete _niz;
	 * 
	 * Na ovaj nacin ce se osloboditi samo prvi element niza, ali ne i celokupan niz.
	 * 
	 * Primetimo da smo niz alocirali naredbom
	 * _niz = new niz[n];
	 * 
	 * Zbog toga, niz moramo da oslobodimo pomocu poziva koji ce primeniti delete na svaki
	 * element niza. To se postize sledecim pozivom:
	 * 
	 * delete[] niz;
	 * 
	 * Dakle, ako new koristimo sa [], moramo na isti nacin da koristimo i delete. 
	 */

	/* stampamo poruku da bismo mogli da pratimo opseg zivota objekta */
	std::cout << "Dinamicki niz: Unistavam objekat" << std::endl;
}

/* metod stampa niz na ostream */
void DinamickiNiz::show(std::ostream& s) const {

	for (int i = 0; i < _size; i++) {
		s << _niz[i] << " ";
	}
}

/* get metod za velicinu niza */
int DinamickiNiz::size() const {

	return _size;
}

/* metod dodaje element na kraj niza 
 * metod menja stanje klase, pa ne moze biti obelezen sa const 
 */
void DinamickiNiz::dodajNaKraj(double x) {

	/* ako nema vise mesta u nize, moramo da realociramo */
	if (_size == _capacity) {

		/* uvecavamo kapacitet niza */
		_capacity *= DEFAULT_STEP;

		/* kreiramo novi duplo veci niz */
		double* noviNiz = new double[_capacity];
		/* kopiramo sadrzaj niza u novi niz */
		for (int i = 0; i < _size; i++)
			noviNiz[i] = _niz[i];

		/* oslobadjamo stari niz */
		delete[] _niz;
		/* i dodeljujemo pokazivac nizu u klasi */
		_niz = noviNiz;
	}

	/* upisujemo element u niz */
	_niz[_size++] = x;
}

/* metod brise element niza sa kraja 
 * metod menja stanje klase, pa ne moze biti obelezen sa const 
 */
void DinamickiNiz::obrisiSaKraja() {

	if (_size == 0)
		return;

	_size -= 1;
}

/* metod ispituje da li se element x nalazi u nizu
 * metod ne menja stanje klase, pa ga obelezavamo sa const 
 */
int DinamickiNiz::pronadji(double x) const {

	for (int i = 0; i < _size; i++) {
		if (_niz[i] == x)
			return i;
	}

	return -1;
}

/* operator indeksiranja koji dozvoljava menjanje sadrzaja niza
 * 
 * BITNO:
 * Obratite paznju na povratnu vrednost operatora. Povratna vrednost je referenca
 * na postojeci element u nizu, jer jedino tako mozemo da promenimo njegovu
 * vrednost. Primetite da se elementi niza nalaze na hipu i da nisu lokalne promenljive
 * sto nam omogucava da kao povratnu vrednost vratimo referencu na element niza, jer ce
 * on svakako postojati nakon sto se izvrsi ova funkcija. 
 */
double& DinamickiNiz::operator [](int i) {

	if (i < 0 || i > _size)
		throw "Index out of bounds";

	return _niz[i];
}

/* const operator indeksiranja koji ne dozvoljava menjanje sadrzaja niza */
const double& DinamickiNiz::operator [](int i) const{

	if (i < 0 || i > _size)
		throw "Index out of bounds";

	return _niz[i];
}

/* globalni operator za stampanje niza */
std::ostream& operator <<(std::ostream& s, const DinamickiNiz& dn) {

	dn.show(s);
	return s;
}
