/* Potrebno je napisati program za rad sa izrazima u analitickom obliku. 
 * U izrazima se mogu pojavljivati:
 * 
 * 1. Konstante - predstavljacemo ih tipom double
 * 2. Promenljive - svaka promenljiva je definisana svojim imenom. u izrazima se moze
 *                  pojavljivati proizvoljan broj promenljivih. 
 * 3. Uobicajene unarne i binarne operacije, kao i funkcije
 *      3.1 Promena znaka (unarni minus)
 *      3.2 Sabiranje, oduzimanje, mnozenje i deljenje
 *      3.3 Funkcije sin, cos, tan, cot, ln, exp, sqrt
 * 4. Radi jednostavnosti, umesto operatora, u internoj reprezentaciji izraza mogu se
 *    koristiti odgovarajuce funkcije. Ako je dat izraz (x-a2)*(y+7.23), mozemo ga zapisati kao
 *    (PUTA (MINUS x a2) (PLUS y 7.23))
 * 
 * Potrebno je obezbediti sledece operacije nad izrazima:
 * 
 * 1. Izracunavanje vrednosti izraza za date vrednosti promenljivih.
 * 2. Uproscavanje izraza zamenjivanjem odredjenog skupa promenljivih datim vrednostima.
 * 3. Osnovne aritmeticke operacije nad izrazima.
 * 4. Izracunavanje izvoda izraza po datoj promenljivoj.
 * 
 * Radi jednostavnosti program prvo ucitava izraz, a zatim skup vrednosti promenljivih.
 * Program treba da izracuna vrednost izraza i da na izlaz ispise izraz uproscen po definisanim promenljivama.
 */
#include <iostream>
#include <map>
#include <exception>
#include <cmath>

/* Da bismo pravilno resili zadatak moramo da sagledamo celine koje postoje u zadatku: 
 *
 * 1. Program radi sa izrazima. Izraz mozemo posmatrati kao entitet definisan pomocu konstanti i promenljivih
 *    potencijalno vezanih nekim operacijama (unarnim, binarnim, funkcijama, itd). Takodje, izraz mogu biti
 *    sacinjeni od nekih podizraza koji su opet povezanim nekim operacijama. 
 * 2. Primetimo da izraz moze da se konkretizuje tek onog trenutka kada su definisane vrednosti promenljivih
 *    koje se u izrazu javljaju. Odavde se moze zakljuciti da vrednost promenljive ne treba da bude deo izraz,
 *    vec treba da bude deo neke tablice simbola. Na slican nacin smo to resavali u parserima koje smo do sada
 *    pisali. 
 * 3. Da bismo vrednosti promenljivih cuvali izdvojeno od izraza, potrebno je da ih na neki zgodan nacin predstavimo
 *    u programu. Dakle, treba da napravimo posebnu klasu, npr Okolina, koja ce enkapsulirati skup promenljivih i 
 *    njima pridruzenih vrednosti, kao i ponasanja koja od klase ocekujemo (definisi promenljivu, obrisi promenljivu,
 *    izmeni promenljivu, saznaj vrednost promenljive itd.)
 * 4. Program treba da omoguci rad sa izrazima. Potrebno je da sagledamo sta sve moze biti izraz i na koji nacin ga mozemo
 *    sastaviti na osnovu postojecih jednostavnijih celina. Na primer, jedan oblik analize moze biti:
 *      4.1. Konstanta moze biti izraz.
 *      4.2. Promenljiva moze biti izraz.
 *      4.3. Izraz moze biti kombinacija vrednosti iz 4.1 i 4.2 povezanih odgovarajucim operatorom (sabiranje, oduzimanje...)
 *      4.4. Izraz moze biti neka unarna transformacija postojecih izraza (promena znaka, primena funkcije itd)
 *      4.5. Izraz moze biti kombinacija izraza 4.1-4.4
 *      4.6. Vrednost izraza definisana je vrednostima promenljivih koje u njemu ucestuvuju.
 *
 * Iz ove male analize mozemo da zakljucimo da nam treba klasa Okolina koja ce cuvati promenljive i njihove vrednosti
 * i u kojoj ce Izraz moci da odredi svoju vrednosti. Takodje, da bismo primenili vecinu operacija nad Izrazima potrebno
 * je da znamo u kojoj Okolini se taj izraz nalazi, pa bi instanca klase Okolina trebalo da bude argument odgovarajucih
 * metoda klase Izraz.
 * 
 * Da bismo omogucili ponasanja iz tacke 4, treba nam hijerarhija klasa Izraz kojom cemo definisati i omoguciti izracunavanje
 * izraza na osnovu postojecih izraza.
 * 
 * Hijerarhija moze biti sledeca: (ovo je sam jedno od mogucih resenja)
 * 
 * Izraz (Apstraktna) ----> Konstanta
 *                 |
 *                 -------> Promenljiva
 *                 |
 *                 -------> BinarniOperator (Apstraktna) ----> Zbir
 *                 |                                     |
 *                 |                                     ----> Razlika
 *                 |                                     |
 *                 |                                     ----> Proizvod
 *                 |                                     |
 *                 |                                     ----> Kolicnik
 *                 |
 *                 -------> Funkcija (Apstraktna) ----> Suprotni (Unarni minus)
 *                                                |
 *                                                ----> Sin
 *                                                |
 *                                                ----> Cos
 *                                                |
 *                                                ----> Tan
 *                                                |
 *                                                ----> Cot
 *                                                |
 *                                                ----> Log
 *                                                |
 *                                                ----> Exp
 *                                                |
 *                                                ----> Sqrt
 *  
 * U programu koji sledi bice ilustrovano kako mozemo da napravimo program za rad sa analitickim izrazima, bez da znamo ista
 * o prevodjenju programskih jezika. Razumevanje ovog primera je kljucno za ono sto nam sledi u nastavku kursa PPJ. Primer je
 * dugacak i kada budete analizirali resenje, uvek imajte na umu sta zelimo da postignemo i pokusajte sa time da povezete
 * resenje. 
 */

/* pomocna klasa u kojoj cuvamo promenljive i njihove vrednosti */
class Okolina {

private:
    /* mapa u kojoj cuvamo imena promenljivih sa pridruzenim vrednostima */
    std::map<std::string, double> _promenljive;

public:
    /* podrazumevani konstruktor */
    Okolina() {

    }

    /* metod dodaje promenljivu u okolinu 
     * const definicija argumenta s je neophodna da bismo prilikom poziva
     * metoda mogli da koristimo konstante. Npr.
     * Okolina o;
     * o.DodajPromenljivu("x", 10);
     * 
     * Kada biste izostavili const, ovakav poziv ne bi bio moguc preko reference, vec biste morali
     * da koristite prenos parametera po vrednosti. 
     */
    bool DodajPromenljivu(const std::string& s, double v) {

        if (PromenljivaDefinisana(s))
            return false;
        
        _promenljive[s] = v;
        return true;
    }

    /* metod proverava da li je promenljiva definisana */
    bool PromenljivaDefinisana(const std::string& s) const {
        
        auto it = _promenljive.find(s);

        return it != _promenljive.end();
    }

    /* metod vraca vrednost promenljive */
    double VrednostPromenljive(const std::string& s) const {

        /* ako promenljiva nije definisana, baca se izuzetak */
        if (!PromenljivaDefinisana(s))
            throw std::invalid_argument("promenljiva nije definisana");
        
        /* inace se vraca vrednost promenljive */
        auto it = _promenljive.find(s);
        return it->second;
    }

    /* metod menja vrednost promenljive */
    void IzmeniVrednostPromenljive(const std::string& s, double v) {

        /* ako promenljiva nije definisana, baca se izuzetak */
        if (!PromenljivaDefinisana(s))
            throw std::invalid_argument("promenljiva nije definisana");
        
        /* menjamo vrednost promenljive */
        _promenljive[s] = v;
    }

    /* operator indeksiranja 
     * ovako definisan operator dozvolajva izmenu vrednosti postojece promenljive
     * u okolini
     */
    double& operator [](std::string& s) {

        /* ako promenljiva nije izuzetak, bacamo izuzetak */
        if (!PromenljivaDefinisana(s))
            throw std::invalid_argument("promenljiva nije definisana");
        
        /* pronalazimo promenljivu u mapi */
        auto it = _promenljive.find(s);
        /* vracamo njenu vrednost */
        return it->second;
    }

    /* const operator indeksiranja */
    double operator [](std::string& s) const{

        if (!PromenljivaDefinisana(s))
            throw std::invalid_argument("promenljiva nije definisana");
        
        auto it = _promenljive.find(s);
        return it->second;
    }
};

/* definisemo hijerarhiju klasa Izraz */
class Izraz {
/* klasa definise samo interfejs 
 * preciznije, klasa nema atribute. 
 */
public:
    /* imamo hijerarhiju klasa, pa moramo da uvezemo destruktore */
    virtual ~Izraz() {
        /* prazna implementacija, jer nema resursa koje treba da unistimo */
    }

    /* metod koji odredjuje vrednost izraza u okolini o */
    virtual double Vrednost(const Okolina& o) const = 0;

    /* metod koji stampa izraz na ostream */
    virtual void Ispisi(std::ostream& s) const = 0;

    /* metod kreira kopiju postojeceg objekta */
    virtual Izraz* Kopija() const = 0;

    /* metod koji u izrazu menja svako pojavljivanje promenljive s njenom
     * vrednosti u okolini o, ako je promenljiva definisana
     */
    virtual Izraz* Uprosti(const Okolina& o, const std::string& s) const = 0;

    /* metod koji racuna izvod po promenljivoj s */
    virtual Izraz* Izvod(const Okolina& o, const std::string& s) const = 0;
};

/* preopterecivanje operatora ispisa */
std::ostream& operator <<(std::ostream& s, const Izraz& i) {
    
    i.Ispisi(s);
    return s;
}

/* definisemo izvedenu klasu konstanta */
class Konstanta : public Izraz {

private: 
    /* konstanta je opisana svojom vrednoscu */
    double _vrednost;

public:
    /* konstruktor */
    Konstanta(double v) {
        _vrednost = v;
    }

    /* klasa je list u hijerarhiji, pa definisemo konstruktor kopije */
    Konstanta(const Konstanta& k) {
        
        _vrednost = k._vrednost;
    }

    /* predefinisemo apstraktne metode */

    /* metod odredjuje vrednost izraza u okolini o */
    double Vrednost(const Okolina& o) const {
        /* vrednost konstante ne zavisi od okoline */
        return _vrednost;
    }

    /* metod ispisuje izraz na izlazni tok s */
    void Ispisi(std::ostream& s) const {
        /* konstanta je predstavljena sobom */
        s << _vrednost;
    }

    /* metod kreira kopiju postojeceg objekta */
    Izraz* Kopija() const {

        /* pozivamo konstruktor kopije i kreiramo kopiju postojeceg izraza */
        return new Konstanta(*this);
    }

    /* metod koji u izrazu menja svako pojavljivanje promenljive s njenom
     * vrednosti u okolini o, ako je promenljiva definisana
     */
    virtual Izraz* Uprosti(const Okolina& o, const std::string& s) const {

        /* konstanta se ne moze dalje uprostiti, pa samo vracamo njenu kopiju */
        return new Konstanta(*this);
    }

    /* metod koji racuna izvod po promenljivoj s */
    virtual Izraz* Izvod(const Okolina& o, const std::string& s) const {

        /* izvod konstante je nova konstanta sa vrednoscu 0 */
        return new Konstanta(0);
    }
};

/* definisemo izvedenu klasu promenljiva */
class Promenljiva : public Izraz {

private:
    /* promenljiva je opisana svojim nazivom */
    std::string _ime;

public:
    /* kosntruktor prihvata samo naziv promenljive  */
    Promenljiva(const std::string& s) {
        _ime = s;
    }

    /* klasa je list u hijerarhiji, pa definisemo konstruktor kopije */
    Promenljiva(const Promenljiva& p) {

        _ime = p._ime;
    }

    /* predefinisemo apstraktne metode */

    /* metod odredjuje vrednost izraza u okolini o */
    double Vrednost(const Okolina& o) const {
        
        /* ako promenljiva nije definisana, bacamo izuzetak */
        if (!o.PromenljivaDefinisana(_ime))
            throw std::invalid_argument("Promenljiva nije definisana");

        /* inace, vracamo vrednost promenljive iz okoline */
        return o.VrednostPromenljive(_ime);
    }

    /* metod koji stampa izraz na ostream */
    void Ispisi(std::ostream& s) const {
        s << _ime;
    }

     /* metod kreira kopiju postojeceg objekta */
    Izraz* Kopija() const {

        return new Promenljiva(*this);
    }

    /* metod koji u izrazu menja svako pojavljivanje promenljive s njenom
     * vrednosti u okolini o, ako je promenljiva definisana
     */
    virtual Izraz* Uprosti(const Okolina& o, const std::string& s) const {

        /* ako je promenljiva po kojoj radimo uproscavanje bas ova 
         * i ako je definisana u okolini
         */
        if (s == _ime && o.PromenljivaDefinisana(s)) {
            /* tada je uproscavamo na konstantu koja kao vrednost ima 
             * bas vrednost promenljive u datoj okolini
             */
            return new Konstanta(o.VrednostPromenljive(s));
        }
        /* ako promenljiva nije definisana u okolini ili se po njoj ne radi 
         * uproscavanje 
         */
        else {
            /* tada je rezultat kopija postojece promenljive 
             * razlog zasto se radi sa kopijama je identican kao u slucaju parsera
             * koje smo do sada pisali  
             */
            return new Promenljiva(*this);
        }
    }

    /* metod koji racuna izvod po promenljivoj s */
    virtual Izraz* Izvod(const Okolina& o, const std::string& s) const {
        
        /* ako se izvod radi bas po promenljivoj */
        if (s == _ime) {
            /* tada je rezultat konstanta 1 */
            return new Konstanta(1);
        }
        else {
            /* inace je rezultat konstanta 0 */
            return new Konstanta(0);
        }
    }
};

/* definisemo izvedenu klasu binarni operator 
 * iz koje cemo kasnije izvesti klase zbir, razlika, proizvod itd
 * na ovom mestu je smisleno zajednicke osobine binarnih operacija apstrahovati u 
 * nadklasu, cime se omogucava dalje lako prosirivanje hijerarhije. 
 */
class BinarniOperator : public Izraz {

protected:
    /* binarni je opisan svojim levim i desnim operandom 
     * koji mogu biti bilo kakvi izrazi
     */
    Izraz *levi, *desni;
    /* Da bismo mogli da koristimo dinamicki polimorfizam, izraze 
     * moramo da cuvamo kao pokazivace. Nivo zastite mora biti protected, 
     * da bi izrazi bili vidljivi u izvedenim klasama
     */
public:
    /* konstruktor klase binarni operator
     * primetite da kosntruktor radi samo kopiranje pokazivace, 
     * ali ne i onoga na sta ti pokazivaci pokazuju (plitko kopiranje)
     */
    BinarniOperator(Izraz* l, Izraz* d) : levi(l), desni(d) {
        
    }
    /* nasa klasa koristi dinamicke resurse, pa je neophodno da napisemo
     * eksplicitni destruktor
     * destruktor u korenu hijerarhije, tj. u klasi Izraz, je obelezen kao virtuelni,
     * cime je omoguceno ispravno nadovezivanje destruktora
     */
    ~BinarniOperator() {
        /* unistavamo resurse iz klase */
        delete levi;
        delete desni;
    }

    /* klasa kao argumente ima dva atributa koji su dinamicki resursi, pa ima smisla da
     * napisemo eksplicitni konstruktor kopije, kojim cemo osigurati duboko kopiranje resursa
     * koje klasa BinarniOperator koristi. 
     */
    BinarniOperator(const BinarniOperator& b) {

        levi = b.levi->Kopija();
        desni = b.desni->Kopija();
    }

    /* bazna klasa nema sopstvenih atributa niti ce izvedene klase imati sopstvenih atributa, 
     * pa u ovom slucaju mozemo da implementiramo operator dodele. 
     * 
     * BITNO:
     * Iako ga mozemo implementirati, to ne znaci da i treba to da cinimo u nadklasi. 
     * Implementiranje operatora dodele na ovaj nacin dozvoljava da kasnije u kodu, na primer, razlici
     * dodelimo zbir. Ova dodela, iako omogucena operatorom dodele u nadklasi BinarniOperator, nije semanticki 
     * ispravna i ne treba biti deo naseg koda. Dakle, ukoliko nam treba dodela, klase u listovima treba da 
     * implementiraju sopstvene (nevirtuelne) operatore dodele, da bi se osigurala semanticka ispravnost
     * operacije dodeljivanja. 
     */
    // BinarniOperator& operator =(const BinarniOperator& z) {

    //     /* self-assignement check*/
    //     if (this == &z) {
    //         return *this;
    //     }

    //     /* unistavamo stare izraze */
    //     delete levi;
    //     delete desni;

    //     /* postavljamo njihove nove vrednosti */
    //     levi = z.levi->Kopija();
    //     desni = z.desni->Kopija();

    //     /* vracamo izmenjeni objekat */
    //     return *this;
    // }

    /* u ovom trenutku i dalje ne znamo o kom operatoru se radi
     * pa metode ostaju apstraktne, jer operacije nisu do kraja definisane
     */

    /* metod odredjuje vrednost izraza u okolini o */
    virtual double Vrednost(const Okolina& a) const = 0;

    /* metod koji stampa izraz na ostream */
    virtual void Ispisi(std::ostream& s) const = 0;

    /* metod kreira kopiju postojeceg objekta */
    virtual Izraz* Kopija() const = 0;

    /* metod koji u izrazu menja svako pojavljivanje promenljive s njenom
     * vrednosti u okolini o, ako je promenljiva definisana
     */
    virtual Izraz* Uprosti(const Okolina& o, const std::string& s) const = 0;

    /* metod koji racuna izvod po promenljivoj s */
    virtual Izraz* Izvod(const Okolina& o, const std::string& s) const = 0;
};

/* definisemo izvedenu klasu Zbir */
class Zbir : public BinarniOperator {

public: 

    /* konstruktor - pozivamo konstruktor bazne klase */
    Zbir(Izraz* l, Izraz* d) : BinarniOperator(l, d) {

    }

    /* konstruktor kopije - zovemo konstruktor kopije bazne klase 
     * koji ce izvrsiti duboko kopiranje atributa 
     */
    Zbir(const Zbir& z) : BinarniOperator(z){

    }
  

    /* implementiramo apstraktne metode */

    /* metod odredjuje vrednost izraza u okolini o */
    double Vrednost(const Okolina& o) const {
        /* metod vrednost moze baciti izuzetak u slucaju da neka promenljiva
         * iz izraza nije definisana, pa moramo da koristimo try-catch blok
         */
        try {
            return levi->Vrednost(o) + desni->Vrednost(o);
        }
        catch (std::invalid_argument& s){
            /* ako se desi izuzetaksamo ce ga proslediti pozivajucoj klasi */
            throw s;
        }
    }

    /* metod koji stampa izraz na ostream */
    void Ispisi(std::ostream& s) const {

        /* izraz predstavljamo u prefiksnoj notaciji */
        s << "( PLUS " << (*levi) << " " << (*desni) << ") ";
    }

     /* metod kreira kopiju postojeceg objekta */
    Izraz* Kopija() const {

        /* kao rezultat, vracamo kopiju tekuceg objekta */
        return new Zbir(*this);
    }

    /* metod koji u izrazu menja svako pojavljivanje promenljive s njenom
     * vrednosti u okolini o, ako je promenljiva definisana
     */
    Izraz* Uprosti(const Okolina& o, const std::string& s) const {

        /* uproscavamo levu i desnu stranu izraza */
        Izraz* l = levi->Uprosti(o, s);
        Izraz* d = desni->Uprosti(o, s);

        /* ako su i leva i desna strana konstante */
        if (dynamic_cast<Konstanta*>(l) != nullptr && 
                dynamic_cast<Konstanta*>(d) != nullptr) {
            /* tada je rezultat uproscavanja nova konstanta */
            Izraz* rez = new Konstanta(l->Vrednost(o) + d->Vrednost(o));
            /* unistavamo polazne objekte */
            delete l; 
            delete d;
            /* i vracamo rezultat */
            return rez;
        }
        /* ako se ne radi o konstantama */
        else {
            /* tada je rezultat novi uprosceni zbir */
            return new Zbir(l,d);
            /* Napomena:
             * Ovde bi mogla da se uradi analiza dobijenih objekata i da se izraz dodatno uprosti
             * Na primer, mozemo proveriti da li je leva ili desna strana izraza konstanta nula, pa
             * svesti izraz samo na jedan operand i tako dalje. 
             */
        }
    }

    /* metod koji racuna izvod po promenljivoj s */
    Izraz* Izvod(const Okolina& o, const std::string& s) const {

        /* izvod zbira jer zbir izvoda, pa zato prvo racunamo izvod leve i desne strane */
        Izraz* l = levi->Izvod(o, s);
        Izraz* d = desni->Izvod(o, s);

        /* provera identicna onoj kod uproscavanja */
        if (dynamic_cast<Konstanta*>(l) != nullptr && 
                dynamic_cast<Konstanta*>(d) != nullptr) {
            Izraz* rez = new Konstanta(l->Vrednost(o) + d->Vrednost(o));
            delete l; 
            delete d;
            return rez;
        }
        else {
            /* ako leva i desna strana nisu konstante, kao rezultat vracamo zbir izvoda */
            return new Zbir(l,d);
        }
    }
};

/* definisemo izvedenu klasu Razlika */
class Razlika : public BinarniOperator {

public: 

    /* konstruktor - pozivamo konstruktor bazne klase */
    Razlika(Izraz* l, Izraz* d) : BinarniOperator(l, d) {

    }

    /* konstruktor kopije - zovemo konstruktor kopije bazne klase 
     * koji ce izvrsiti duboko kopiranje atributa 
     */
    Razlika(const Razlika& z) : BinarniOperator(z){

    }
  

    /* implementiramo apstraktne metode */

    /* metod odredjuje vrednost izraza u okolini o */
    double Vrednost(const Okolina& o) const {
        /* metod vrednost moze baciti izuzetak u slucaju da neka promenljiva
         * iz izraza nije definisana, pa moramo da koristimo try-catch blok
         */
        try {
            return levi->Vrednost(o) - desni->Vrednost(o);
        }
        catch (std::invalid_argument& s){
            /* ako se desi izuzetaksamo ce ga proslediti pozivajucoj klasi */
            throw s;
        }
    }

    /* metod koji stampa izraz na ostream */
    void Ispisi(std::ostream& s) const {

        /* izraz predstavljamo u prefiksnoj notaciji */
        s << "( MINUS " << (*levi) << " " << (*desni) << ") ";
    }

     /* metod kreira kopiju postojeceg objekta */
    Izraz* Kopija() const  {

        /* kao rezultat, vracamo kopiju tekuceg objekta */
        return new Razlika(*this);
    }

    /* metod koji u izrazu menja svako pojavljivanje promenljive s njenom
     * vrednosti u okolini o, ako je promenljiva definisana
     */
    Izraz* Uprosti(const Okolina& o, const std::string& s) const {

        /* postupak identican kao kod zbira */
        Izraz* l = levi->Uprosti(o, s);
        Izraz* d = desni->Uprosti(o, s);

        if (dynamic_cast<Konstanta*>(l) != nullptr && 
                dynamic_cast<Konstanta*>(d) != nullptr) {
            Izraz* rez = new Konstanta(l->Vrednost(o) - d->Vrednost(o));
            delete l; 
            delete d;
            return rez;
        }
        else 
            return new Razlika(l,d);
    }

    /* metod koji racuna izvod po promenljivoj s */
    Izraz* Izvod(const Okolina& o, const std::string& s) const {

        /* postupak identican kao kod zbira */
        Izraz* l = levi->Uprosti(o, s);
        Izraz* d = desni->Uprosti(o, s);

        if (dynamic_cast<Konstanta*>(l) != nullptr && 
                dynamic_cast<Konstanta*>(d) != nullptr) {
            Izraz* rez = new Konstanta(l->Vrednost(o) - d->Vrednost(o));
            delete l; 
            delete d;
            return rez;
        }
        else 
            return new Razlika(l,d);
    }
};

/* definisemo izvedenu klasu Proizvod */
class Proizvod : public BinarniOperator {

public: 

    /* konstruktor - pozivamo konstruktor bazne klase */
    Proizvod(Izraz* l, Izraz* d) : BinarniOperator(l, d) {

    }

    /* konstruktor kopije - zovemo konstruktor kopije bazne klase 
     * koji ce izvrsiti duboko kopiranje atributa 
     */
    Proizvod(const Razlika& z) : BinarniOperator(z){

    }
  

    /* implementiramo apstraktne metode */

    /* metod odredjuje vrednost izraza u okolini o */
    double Vrednost(const Okolina& o) const {
        /* metod vrednost moze baciti izuzetak u slucaju da neka promenljiva
         * iz izraza nije definisana, pa moramo da koristimo try-catch blok
         */
        try {
            return levi->Vrednost(o) * desni->Vrednost(o);
        }
        catch (std::invalid_argument& s){
            /* ako se desi izuzetaksamo ce ga proslediti pozivajucoj klasi */
            throw s;
        }
    }

    /* metod koji stampa izraz na ostream */
    void Ispisi(std::ostream& s) const {

        /* izraz predstavljamo u prefiksnoj notaciji */
        s << "( PUTA " << (*levi) << " " << (*desni) << ") ";
    }

     /* metod kreira kopiju postojeceg objekta */
    Izraz* Kopija() const  {

        /* kao rezultat, vracamo kopiju tekuceg objekta */
        return new Proizvod(*this);
    }

    /* metod koji u izrazu menja svako pojavljivanje promenljive s njenom
     * vrednosti u okolini o, ako je promenljiva definisana
     */
    Izraz* Uprosti(const Okolina& o, const std::string& s) const  {

        /* postupak identican kao kod zbira */
        Izraz* l = levi->Uprosti(o, s);
        Izraz* d = desni->Uprosti(o, s);

        if (dynamic_cast<Konstanta*>(l) != nullptr && 
                dynamic_cast<Konstanta*>(d) != nullptr) {
            Izraz* rez = new Konstanta(l->Vrednost(o) * d->Vrednost(o));
            delete l; 
            delete d;
            return rez;
        }
        else 
            return new Proizvod(l,d);
    }

    /* metod koji racuna izvod po promenljivoj s */
    Izraz* Izvod(const Okolina& o, const std::string& s) const {

        /* primenjujemo formulu za izvod proizvoda */

        /* prvo odredjujemo izvode levog i desnog operanda */
        Izraz *il = levi->Izvod(o, s);
        Izraz *id = desni->Izvod(o, s);

        /* zatim formiramo levi i desni operand konacnog rezultat 
         * primetite da ovde koristimo kopije originalnih atributa, da bismo
         * osigurali da svaki novi izraz ima svoju kopiju izraza od kojih je sacinjen
         * u suprotnom, mozemo doci u situaciju u kojoj program nije u konzistentnom stanju.
         */
        Izraz* ll = new Proizvod(il, desni->Kopija());
        Izraz* dd = new Proizvod(levi->Kopija(), id);

        /* rezultat je zbir levog i desnog operanda */
        return new Zbir(ll, dd);
    }
};

/* definisemo izvedenu klasu Kolicnik */
class Kolicnik : public BinarniOperator {

public: 

    /* konstruktor - pozivamo konstruktor bazne klase */
    Kolicnik(Izraz* l, Izraz* d) : BinarniOperator(l, d) {

    }

    /* konstruktor kopije - zovemo konstruktor kopije bazne klase 
     * koji ce izvrsiti duboko kopiranje atributa 
     */
    Kolicnik(const Razlika& z) : BinarniOperator(z){

    }
  

    /* implementiramo apstraktne metode */

    /* metod odredjuje vrednost izraza u okolini o */
    double Vrednost(const Okolina& o) const {
        if (desni->Vrednost(o) != 0)
            return levi->Vrednost(o) / desni->Vrednost(o);
        else 
            throw std::invalid_argument("Deljenje nulom");
    }

    /* metod koji stampa izraz na ostream */
    void Ispisi(std::ostream& s) const {

        /* izraz predstavljamo u prefiksnoj notaciji */
        s << "( PODELJENO " << (*levi) << " " << (*desni) << ") ";
    }

     /* metod kreira kopiju postojeceg objekta */
    Izraz* Kopija() const  {

        /* kao rezultat, vracamo kopiju tekuceg objekta */
        return new Kolicnik(*this);
    }

    /* metod koji u izrazu menja svako pojavljivanje promenljive s njenom
     * vrednosti u okolini o, ako je promenljiva definisana
     */
    Izraz* Uprosti(const Okolina& o, const std::string& s) const  {
        
        /* postupak identican kao kod zbira */
        Izraz* l = levi->Uprosti(o, s);
        Izraz* d = desni->Uprosti(o, s);

        if (dynamic_cast<Konstanta*>(l) != nullptr && 
                dynamic_cast<Konstanta*>(d) != nullptr) {
            Izraz* rez = new Konstanta(l->Vrednost(o) / d->Vrednost(o));
            delete l; 
            delete d;
            return rez;
        }
        else 
            return new Kolicnik(l,d);
    }

    /* metod koji racuna izvod po promenljivoj s */
    Izraz* Izvod(const Okolina& o, const std::string& s) const {

        /* primenjujemo formulu za izvod kolicnika */

        /* odredjuemo izvode polaznih operanada */
        Izraz *il = levi->Izvod(o, s);
        Izraz *id = desni->Izvod(o, s);

        /* racunamo levi i desni operand brojioca rezultata 
         * primetimo da i ovde radimo sa kopijama iz istog razloga kao i kod
         * zbira. 
         */
        Izraz* ll = new Proizvod(il, desni->Kopija());
        Izraz* dd = new Proizvod(levi->Kopija(), id);

        /* racunamo brojilac i imenilac rezultat */
        Izraz* gore = new Razlika(ll, dd);
        Izraz* dole = new Proizvod(desni->Kopija(), desni->Kopija());

        /* rezultat je kolicnik dobijenih vrednosti */
        return new Kolicnik(gore, dole);
    }
};

/* izvodimo klasu za reprezentaciju funkcija, jer funkcije mozemo na neki nacin 
 * posmatrati kao unarne operatore. Primenjujemo slicno razmisljanje kao 
 * za Binarne operatore. 
 */
class Funkcija : public Izraz {

protected:
    /* argument funkcije je jedan izraz, tj. onaj na koji treba primeniti funkciju */
    Izraz* izraz;
public:
    /* konstruktor */
    Funkcija(Izraz* i) : izraz(i) {

    }
    /* konstruktor kopije */
    Funkcija(const Funkcija& f) {

        izraz = f.izraz->Kopija();
    }
    /* virtuelni destruktor, jer se radi o hijerarhiji klasa */
    virtual ~Funkcija() {
        delete izraz;
    }

    /* bazna klasa nema sopstvenih atributa niti ce izvedene klase imati sopstvenih atributa, 
     * pa u ovom slucaju mozemo da implementiramo operator dodele. 
     * 
     * BITNO:
     * Iako ga mozemo implementirati, to ne znaci da i treba to da cinimo u nadklasi. 
     * Implementiranje operatora dodele na ovaj nacin dozvoljava da kasnije u kodu, na primer, funkciji sinus
     * dodelimo funkciju kosinus. Ova dodela, iako omogucena operatorom dodele u nadklasi UnarniOperator, nije semanticki 
     * ispravna i ne treba biti deo naseg koda. Dakle, ukoliko nam treba dodela, klase u listovima treba da 
     * implementiraju sopstvene (nevirtuelne) operatore dodele, da bi se osigurala semanticka ispravnost
     * operacije dodeljivanja. 
     */
    // Funkcija& operator =(const Funkcija& f) {

    //     /* self assignment check */
    //     if (this == &f) {
    //         return *this;
    //     }

    //     /* unistavamo stari izraz */
    //     delete izraz;
    //     /* kopiramo izraz iz argumenta */
    //     izraz = f.izraz->Kopija();
    //     /* vracamo izmenjeni objekat */
    //     return *this;
    // }

    /* apstraktne metode */

    /* metod odredjuje vrednost izraza u okolini o */
    virtual double Vrednost(const Okolina& a) const = 0;

    /* metod koji stampa izraz na ostream */
    virtual void Ispisi(std::ostream& s) const = 0;

    /* metod kreira kopiju postojeceg objekta */
    virtual Izraz* Kopija() const = 0;

    /* metod koji u izrazu menja svako pojavljivanje promenljive s njenom
     * vrednosti u okolini o, ako je promenljiva definisana
     */
    virtual Izraz* Uprosti(const Okolina& o, const std::string& s) const = 0;

    /* metod koji racuna izvod po promenljivoj s */
    virtual Izraz* Izvod(const Okolina& o, const std::string& s) const = 0;
};

/* implementiramo unarni minus */
class Suprotni : public Funkcija {

public:
    /* konstruktor */
    Suprotni(Izraz* i) : Funkcija(i) {

    }
    Suprotni(const Suprotni& s) : Funkcija(s) {

    }

    /* metod odredjuje vrednost izraza u okolini o */
    double Vrednost(const Okolina& a) const {

        return -izraz->Vrednost(a);
    }

    /* metod koji stampa izraz na ostream */
    void Ispisi(std::ostream& s) const {
        s << "SUPROTNI ( " << (*izraz) << ") "; 
    }

    /* metod kreira kopiju postojeceg objekta */
    Izraz* Kopija() const {
        
        return new Suprotni(*this);
    }

     /* metod kreira kopiju postojeceg objekta */
    Izraz* Uprosti(const Okolina& o, const std::string& s) const {

        Izraz* l = izraz->Uprosti(o, s);

        if (dynamic_cast<Konstanta*>(l) != nullptr) {
            Izraz* rez = new Konstanta(-l->Vrednost(o));
            delete l;
            return rez;
        }
        else 
            return new Suprotni(l);
    }

    /* metod koji u izrazu menja svako pojavljivanje promenljive s njenom
     * vrednosti u okolini o, ako je promenljiva definisana
     */
    Izraz* Izvod(const Okolina& o, const std::string& s) const {

        Izraz* l = izraz->Izvod(o, s);
        if (dynamic_cast<Konstanta*>(l) != nullptr) {
            Izraz* rez = new Konstanta(-l->Vrednost(o));
            delete l;
            return rez;
        }
        else 
            return new Suprotni(l);
    }
};

class Sin : public Funkcija {

public:
    /* konstruktor */
    Sin(Izraz* i) : Funkcija(i) {

    }
    Sin(const Sin& s) : Funkcija(s) {

    }

    /* metod odredjuje vrednost izraza u okolini o */
    double Vrednost(const Okolina& a) const {

        return sin(-izraz->Vrednost(a));
    }

    /* metod koji stampa izraz na ostream */
    void Ispisi(std::ostream& s) const {
        s << "SIN ( " << (*izraz) << ") "; 
    }

    /* metod kreira kopiju postojeceg objekta */
    Izraz* Kopija() const {
        
        return new Sin(*this);
    }

    /* metod koji u izrazu menja svako pojavljivanje promenljive s njenom
     * vrednosti u okolini o, ako je promenljiva definisana
     */
    Izraz* Uprosti(const Okolina& o, const std::string& s) const {

        Izraz* l = izraz->Uprosti(o, s);

        if (dynamic_cast<Konstanta*>(l) != nullptr) {
            Izraz* rez = new Konstanta(sin(l->Vrednost(o)));
            delete l;
            return rez;
        }
        else 
            return new Sin(l);
    }

    /* metod koji racuna izvod po promenljivoj s */
    Izraz* Izvod(const Okolina& o, const std::string& s) const;
};

class Cos : public Funkcija {

public:
    /* konstruktor */
    Cos(Izraz* i) : Funkcija(i) {

    }
    Cos(const Cos& s) : Funkcija(s) {

    }

    /* metod odredjuje vrednost izraza u okolini o */
    double Vrednost(const Okolina& a) const {

        return cos(-izraz->Vrednost(a));
    }

    /* metod koji stampa izraz na ostream */
    void Ispisi(std::ostream& s) const {
        s << "COS ( " << (*izraz) << ") "; 
    }

    /* metod kreira kopiju postojeceg objekta */
    Izraz* Kopija() const {
        
        return new Cos(*this);
    }

    /* metod koji u izrazu menja svako pojavljivanje promenljive s njenom
     * vrednosti u okolini o, ako je promenljiva definisana
     */
    Izraz* Uprosti(const Okolina& o, const std::string& s) const {

        Izraz* l = izraz->Uprosti(o, s);

        if (dynamic_cast<Konstanta*>(l) != nullptr) {
            Izraz* rez = new Konstanta(cos(l->Vrednost(o)));
            delete l;
            return rez;
        }
        else 
            return new Cos(l);
    }

    /* metod koji racuna izvod po promenljivoj s */
    Izraz* Izvod(const Okolina& o, const std::string& s) const {

        Izraz* l = izraz->Izvod(o, s);
        
        return new Suprotni(new Proizvod(new Sin(izraz->Kopija()), l));
    }
};

Izraz* Sin::Izvod(const Okolina& o, const std::string& s) const {

        Izraz* l = izraz->Izvod(o, s);
        
        return new Proizvod(new Cos(izraz->Kopija()), l);
    }

class Tan : public Funkcija {

public:
    /* konstruktor */
    Tan(Izraz* i) : Funkcija(i) {

    }
    Tan(const Tan& s) : Funkcija(s) {

    }

    /* metod odredjuje vrednost izraza u okolini o */
    double Vrednost(const Okolina& a) const {

        return tan(-izraz->Vrednost(a));
    }

    /* metod koji stampa izraz na ostream */
    void Ispisi(std::ostream& s) const {
        s << "TAN ( " << (*izraz) << ") "; 
    }

    /* metod kreira kopiju postojeceg objekta */
    Izraz* Kopija() const {
        
        return new Tan(*this);
    }

    /* metod koji u izrazu menja svako pojavljivanje promenljive s njenom
     * vrednosti u okolini o, ako je promenljiva definisana
     */
    Izraz* Uprosti(const Okolina& o, const std::string& s) const {

        Izraz* l = izraz->Uprosti(o, s);

        if (dynamic_cast<Konstanta*>(l) != nullptr) {
            Izraz* rez = new Konstanta(tan(l->Vrednost(o)));
            delete l;
            return rez;
        }
        else 
            return new Tan(l);
    }

    /* metod koji racuna izvod po promenljivoj s */
    Izraz* Izvod(const Okolina& o, const std::string& s) const {

        Izraz* l = izraz->Izvod(o, s);
        
        return new Kolicnik(l, new Proizvod(new Cos(izraz->Kopija()), new Cos(izraz->Kopija())));
    }
};

class Cot : public Funkcija {

public:
    /* konstruktor */
    Cot(Izraz* i) : Funkcija(i) {

    }
    Cot(const Cot& s) : Funkcija(s) {

    }

    /* metod odredjuje vrednost izraza u okolini o */
    double Vrednost(const Okolina& a) const {
        
        double x = -izraz->Vrednost(a);
        return cos(x)/sin(x);
    }

    /* metod koji stampa izraz na ostream */
    void Ispisi(std::ostream& s) const {
        s << "COT ( " << (*izraz) << ") "; 
    }

    /* metod kreira kopiju postojeceg objekta */
    Izraz* Kopija() const {
        
        return new Cot(*this);
    }

    /* metod koji u izrazu menja svako pojavljivanje promenljive s njenom
     * vrednosti u okolini o, ako je promenljiva definisana
     */
    Izraz* Uprosti(const Okolina& o, const std::string& s) const {

        Izraz* l = izraz->Uprosti(o, s);

        if (dynamic_cast<Konstanta*>(l) != nullptr) {
            double x = l->Vrednost(o);
            Izraz* rez = new Konstanta(cos(x)/sin(x));
            delete l;
            return rez;
        }
        else 
            return new Cot(l);
    }

    /* metod koji racuna izvod po promenljivoj s */
    Izraz* Izvod(const Okolina& o, const std::string& s) const {

        Izraz* l = izraz->Izvod(o, s);
        
        return new Kolicnik(new Suprotni(l), new Proizvod(new Sin(izraz->Kopija()), new Sin(izraz->Kopija())));
    }
};

class Log : public Funkcija {

public:
    /* konstruktor */
    Log(Izraz* i) : Funkcija(i) {

    }
    Log(const Log& s) : Funkcija(s) {

    }

    /* metod odredjuje vrednost izraza u okolini o */
    double Vrednost(const Okolina& a) const {
        
        return log(izraz->Vrednost(a));
    }

    /* metod koji stampa izraz na ostream */
    void Ispisi(std::ostream& s) const {
        s << "LOG ( " << (*izraz) << ") "; 
    }

    /* metod kreira kopiju postojeceg objekta */
    Izraz* Kopija() const {
        
        return new Log(*this);
    }

    /* metod koji u izrazu menja svako pojavljivanje promenljive s njenom
     * vrednosti u okolini o, ako je promenljiva definisana
     */
    Izraz* Uprosti(const Okolina& o, const std::string& s) const {

        Izraz* l = izraz->Uprosti(o, s);

        if (dynamic_cast<Konstanta*>(l) != nullptr) {
            double x = l->Vrednost(o);
            Izraz* rez = new Konstanta(log(x));
            delete l;
            return rez;
        }
        else 
            return new Log(l);
    }

    /* metod koji racuna izvod po promenljivoj s */
    Izraz* Izvod(const Okolina& o, const std::string& s) const {

        Izraz* l = izraz->Izvod(o, s);
        
        return new Proizvod(new Kolicnik(new Konstanta(1), izraz->Kopija()), l);
    }
};

class Exp : public Funkcija {

public:
    /* konstruktor */
    Exp(Izraz* i) : Funkcija(i) {

    }
    Exp(const Exp& s) : Funkcija(s) {

    }

    /* metod odredjuje vrednost izraza u okolini o */
    double Vrednost(const Okolina& a) const {
        
        return exp(izraz->Vrednost(a));
    }

    /* metod koji stampa izraz na ostream */
    void Ispisi(std::ostream& s) const {
        s << "EXP ( " << (*izraz) << ") "; 
    }

    /* metod kreira kopiju postojeceg objekta */
    Izraz* Kopija() const {
        
        return new Exp(*this);
    }

    /* metod koji u izrazu menja svako pojavljivanje promenljive s njenom
     * vrednosti u okolini o, ako je promenljiva definisana
     */
    Izraz* Uprosti(const Okolina& o, const std::string& s) const {

        Izraz* l = izraz->Uprosti(o, s);

        if (dynamic_cast<Konstanta*>(l) != nullptr) {
            double x = l->Vrednost(o);
            Izraz* rez = new Konstanta(exp(x));
            delete l;
            return rez;
        }
        else 
            return new Log(l);
    }

    Izraz* Izvod(const Okolina& o, const std::string& s) const {

        Izraz* l = izraz->Izvod(o, s);
        
        return new Proizvod(new Exp(izraz->Kopija()), l);
    }
};

class Sqrt : public Funkcija {

public:
    /* konstruktor */
    Sqrt(Izraz* i) : Funkcija(i) {

    }
    Sqrt(const Sqrt& s) : Funkcija(s) {

    }

    /* metod odredjuje vrednost izraza u okolini o */
    double Vrednost(const Okolina& a) const {
        
        return sqrt(izraz->Vrednost(a));
    }

    /* metod koji stampa izraz na ostream */
    void Ispisi(std::ostream& s) const {
        s << "SQRT ( " << (*izraz) << ") "; 
    }

    /* metod kreira kopiju postojeceg objekta */
    Izraz* Kopija() const {
        
        return new Sqrt(*this);
    }

    /* metod koji u izrazu menja svako pojavljivanje promenljive s njenom
     * vrednosti u okolini o, ako je promenljiva definisana
     */
    Izraz* Uprosti(const Okolina& o, const std::string& s) const {

        Izraz* l = izraz->Uprosti(o, s);

        if (dynamic_cast<Konstanta*>(l) != nullptr) {
            double x = l->Vrednost(o);
            Izraz* rez = new Konstanta(sqrt(x));
            delete l;
            return rez;
        }
        else 
            return new Log(l);
    }

    /* metod koji racuna izvod po promenljivoj s */
    Izraz* Izvod(const Okolina& o, const std::string& s) const {

        Izraz* l = izraz->Izvod(o, s);
        
        return new Kolicnik(new Suprotni(l), new Proizvod(new Konstanta(2), new Sqrt(izraz->Kopija())));
    }
};

int main() {

    Okolina o;
    o.DodajPromenljivu("x", 5);
    o.DodajPromenljivu("y", 10);

    // (x/2) + (y - (3.14*z))
    Izraz* i1 = new Kolicnik(new Promenljiva("x"), new Sin(new Konstanta(2)));
    Izraz* i2 = new Proizvod(new Exp(new Sqrt(new Konstanta(3.14))), new Promenljiva("x"));
    Izraz* i3 = new Razlika(new Promenljiva("y"), i2->Kopija());
    Izraz* i4 = new Zbir(i1->Kopija(), i3->Kopija());

    std::cout << (*i4) << "=" << (i4->Vrednost(o)) << std::endl;

    Izraz* izvod = i4->Izvod(o, "x'");
    std::cout << (*izvod) << "=" << (izvod->Vrednost(o)) << std::endl;

    Izraz* i5 = i4->Uprosti(o, "x");
    std::cout << (*i5) << "=" << (i5->Vrednost(o)) << std::endl;

    Izraz* i6 = i5->Uprosti(o, "y");
    std::cout << (*i6) << "=" << (i6->Vrednost(o)) << std::endl;

    /* treba biti obazriv sa brisanjem objekata, jer u klasama ne cuvamo uvek kopije,
     * pa treba pazljivo oslobadjati memoriju.
     */
    delete i5;
    delete i4;
    delete i6;
    delete izvod;

    return 0;
}