#include "SyntaxTree.hpp"

/*************************************** Identifier node *********************************************/
IdentifierNode::IdentifierNode(const std::string& name) {
    _name = name;
}

int IdentifierNode::Interpret(SymbolTable& st) const {
    int result = 0;
    bool retVal = st.GetVariable(_name, &result);

    if (retVal == false)
        yyerror("Promenljiva nije definisana");

    return result;
}
void IdentifierNode::Print() const {
    std::cout << _name;
}
SynTreeNode* IdentifierNode::Clone() const {
    return new IdentifierNode(*this);
}

/*************************************** Const node *********************************************/
ConstNode::ConstNode(int value) {
    _value = value;
}

int ConstNode::Interpret(SymbolTable& st) const {
    return _value;
}
void ConstNode::Print() const {
    std::cout << _value;
}
SynTreeNode* ConstNode::Clone() const {
    return new ConstNode(*this);
}

/*************************************** Group node *********************************************/
GroupNode::GroupNode(SynTreeNode* expr) {
    _expr = expr;
}
GroupNode::GroupNode(const GroupNode& uo) {
    _expr = uo._expr->Clone();
}
GroupNode::~GroupNode() {
    delete _expr;
}

int GroupNode::Interpret(SymbolTable& st) const {
    return _expr->Interpret(st);
}
void GroupNode::Print() const {
    std::cout << "(";
    _expr->Print();
    std::cout << ")";
}
SynTreeNode* GroupNode::Clone() const {
    return new GroupNode(*this);
}

/*************************************** Binary node *********************************************/
BinaryOperationNode::BinaryOperationNode(const std::string& symbol, SynTreeNode* left, SynTreeNode* right) {
    _symbol = symbol;
    _left = left;
    _right = right;
}
BinaryOperationNode::BinaryOperationNode(const BinaryOperationNode& bo) {
    _symbol = bo._symbol;
    _left = bo._left->Clone();
    _right = bo._right->Clone();
}
BinaryOperationNode::~BinaryOperationNode() {
    delete _left;
    delete _right;
}

int BinaryOperationNode::Interpret(SymbolTable& st) const {
    if (_symbol == "+") {
        return _left->Interpret(st) + _right->Interpret(st);
    }
    else if (_symbol == "-") {
        return _left->Interpret(st) - _right->Interpret(st);
    }
    else if (_symbol == "*") {
        return _left->Interpret(st) * _right->Interpret(st);
    }
    else if (_symbol == "/") {
        int delilac = _right->Interpret(st);
        if (delilac == 0)
            yyerror("Deljenje nulom");
        return _left->Interpret(st) / delilac;
    }
    else if (_symbol == "%") {
        int delilac = _right->Interpret(st);
        if (delilac == 0)
            yyerror("Deljenje nulom");
        return _left->Interpret(st) % delilac;
    }
    else if (_symbol == "&") {
        return _left->Interpret(st) & _right->Interpret(st);
    }
    else if (_symbol == "|") {
        return _left->Interpret(st) | _right->Interpret(st);
    }
    else if (_symbol == "^") {
        return _left->Interpret(st) ^ _right->Interpret(st);
    }
    else if (_symbol == "<<") {
        return _left->Interpret(st) << _right->Interpret(st);
    }
    else if (_symbol == ">>") {
        return _left->Interpret(st) >> _right->Interpret(st);
    }
    yyerror("Nepoznata operacija");
    return 0;
}
void BinaryOperationNode::Print() const {
    _left->Print();
    std::cout << " " << _symbol << " ";
    _right->Print();
}
SynTreeNode* BinaryOperationNode::Clone() const {
    return new BinaryOperationNode(*this);
}

/*************************************** Unary node *********************************************/
UnaryOperationNode::UnaryOperationNode(const std::string& symbol, SynTreeNode* expr) {
    _symbol = symbol;
    _expr = expr;
}
UnaryOperationNode::UnaryOperationNode(const UnaryOperationNode& uo) {
    _symbol = uo._symbol;
    _expr = uo._expr->Clone();
}
UnaryOperationNode::~UnaryOperationNode() {
    delete _expr;
}

int UnaryOperationNode::Interpret(SymbolTable& st) const {
    if (_symbol == "-") {
        return -_expr->Interpret(st);
    }
    else if (_symbol == "~") {
        return ~_expr->Interpret(st);
    }
    yyerror("Nepoznata operacija");
    return 0;
}
void UnaryOperationNode::Print() const {
    std::cout << _symbol;
    _expr->Print();
}
SynTreeNode* UnaryOperationNode::Clone() const {
    return new UnaryOperationNode(*this);
}

/*************************************** Logical node *********************************************/
LogicalOperationNode::LogicalOperationNode(const std::string& symbol, SynTreeNode* left, SynTreeNode* right) {
    _symbol = symbol;
    _left = left;
    _right = right;
}
LogicalOperationNode::LogicalOperationNode(const LogicalOperationNode& bo) {
    _symbol = bo._symbol;
    _left = bo._left->Clone();
    _right = bo._right->Clone();
}
LogicalOperationNode::~LogicalOperationNode() {
    delete _left;
    delete _right;
}

int LogicalOperationNode::Interpret(SymbolTable& st) const {
    if (_symbol == "<") {
        return _left->Interpret(st) < _right->Interpret(st);
    }
    else if (_symbol == ">") {
        return _left->Interpret(st) > _right->Interpret(st);
    }
    else if (_symbol == "<=") {
        return _left->Interpret(st) <= _right->Interpret(st);
    }
    else if (_symbol == ">=") {
        return _left->Interpret(st) >= _right->Interpret(st);
    }
    else if (_symbol == "==") {
        return _left->Interpret(st) == _right->Interpret(st);
    }
    else if (_symbol == "!=") {
        return _left->Interpret(st) != _right->Interpret(st);
    }
    yyerror("Nepoznata operacija");
    return 0;
}
void LogicalOperationNode::Print() const {
    _left->Print();
    std::cout << " " << _symbol << " ";
    _right->Print();
}
SynTreeNode* LogicalOperationNode::Clone() const {
    return new LogicalOperationNode(*this);
}

/*************************************** Print node *********************************************/

PrintNode::PrintNode(SynTreeNode* expr, char fmt) {
    _expr = expr;
    _fmt = fmt;
    _logical = false;
}
PrintNode::PrintNode(SynTreeNode* expr, bool logical) {
    _expr = expr;
    _logical = logical;
    _fmt = 'd';
}
PrintNode::PrintNode(const PrintNode& pn) {
    _fmt = pn._fmt;
    _logical = pn._logical;
    _expr = pn._expr->Clone();
}
PrintNode::~PrintNode() {
    delete _expr;
}

int PrintNode::Interpret(SymbolTable& st) const {
    int result = _expr->Interpret(st);
    if (_logical) {
        std::cout << (result ? "True" : "False") << std::endl;
    }
    else {
        if (_fmt == 'd') {
            std::cout << result << std::endl;
        }
        else if (_fmt == 'h') {
            std::cout << std::hex << result << std::dec << std::endl;
        }
        else if (_fmt == 'b') {
            std::cout << std::bitset<8*sizeof(int)>(result) << std::endl;
        }
        else {
            yyerror("Nepoznat format ispisa");
        }
    }
    return 0;
}
void PrintNode::Print() const {
    if (_logical) {
        _expr->Print();
    }
    else {
        std::cout<< "print(\"%" << _fmt << "\", ";
        _expr->Print();
        std::cout << ")";
    }
}
SynTreeNode* PrintNode::Clone() const {
    return new PrintNode(*this);
}

/*************************************** Assignment node *********************************************/
AssignmentNode::AssignmentNode(const std::string& name, SynTreeNode* expr) {
    _name = name;
    _expr = expr;
    _definition = false;
}
AssignmentNode::AssignmentNode(const std::string& name, SynTreeNode* expr, bool def) {
    _name = name;
    _expr = expr;
    _definition = def;
}
AssignmentNode::AssignmentNode(const AssignmentNode& pn){
    _name = pn._name;
    _definition = pn._definition;
    _expr = pn._expr->Clone();
}
AssignmentNode::~AssignmentNode() {
    delete _expr;
}

int AssignmentNode::Interpret(SymbolTable& st) const {
    int result = _expr->Interpret(st);
    if (_definition) {
        bool result = st.DefineVariable(_name, result);
        if (result == false)
            yyerror("Promenljiva vec definisana");
    }
    else {
        bool result = st.UpdateVariable(_name, result);
        if (result == false)
            yyerror("Promenljiva nije definisana");
    }
    return result;
}
void AssignmentNode::Print() const {
    if (_definition)
        std::cout << "def ";
    std::cout << _name << " = ";
    _expr->Print();

}
SynTreeNode* AssignmentNode::Clone() const {
    return new AssignmentNode(*this);
}

/*************************************** Statement Sequence node *********************************************/
StatementSequenceNode::StatementSequenceNode(const StatementSequenceNode& ssn) {
    for (int i = 0; i < ssn._stmts.size(); i++) {
        _stmts.push_back(ssn._stmts[i]->Clone());
    }
}
StatementSequenceNode::~StatementSequenceNode() {
    for (int i = 0; i < _stmts.size(); i++) {
        delete _stmts[i];
    }
}

void StatementSequenceNode::Push(SynTreeNode* stmt) {
    _stmts.push_back(stmt);
}
int StatementSequenceNode::Interpret(SymbolTable& st) const {
    for (int i = 0; i < _stmts.size(); i++) {
        _stmts[i]->Interpret(st);
    }
    return 0;
}
void StatementSequenceNode::Print() const {
    for (int i = 0; i < _stmts.size(); i++) {
        _stmts[i]->Print();
        std::cout << ";" << std::endl;
    }
}
SynTreeNode* StatementSequenceNode::Clone() const {
    return new StatementSequenceNode(*this);
}