#ifndef SYN_TREE_H
#define SYN_TREE_H

#include <iostream>
#include <cstdlib>
#include <bitset>
#include <vector>
#include "SymbolTable.hpp"

extern void yyerror(std::string s);

class SynTreeNode {

public:
    virtual ~SynTreeNode() {}

    virtual int Interpret(SymbolTable& st) const = 0;
    virtual void Print() const = 0;
    virtual SynTreeNode* Clone() const = 0;
};

class IdentifierNode : public SynTreeNode {
private: 
    std::string _name;
public:
    IdentifierNode(const std::string& name);

    virtual int Interpret(SymbolTable& st) const;
    virtual void Print() const;
    virtual SynTreeNode* Clone() const;
};

class ConstNode : public SynTreeNode {
private: 
    int _value;
public:
    ConstNode(int value);

    virtual int Interpret(SymbolTable& st) const;
    virtual void Print() const;
    virtual SynTreeNode* Clone() const;
};

class GroupNode : public SynTreeNode {
private:
    SynTreeNode* _expr;
public:
    GroupNode(SynTreeNode* expr);
    GroupNode(const GroupNode& uo);
    ~GroupNode();

    virtual int Interpret(SymbolTable& st) const;
    virtual void Print() const;
    virtual SynTreeNode* Clone() const;  
};

class BinaryOperationNode : public SynTreeNode {
private:
    std::string _symbol;
    SynTreeNode* _left;
    SynTreeNode* _right;
public:
    BinaryOperationNode(const std::string& symbol, SynTreeNode* left, SynTreeNode* right);
    BinaryOperationNode(const BinaryOperationNode& bo);
    ~BinaryOperationNode();

    virtual int Interpret(SymbolTable& st) const;
    virtual void Print() const;
    virtual SynTreeNode* Clone() const;
};

class UnaryOperationNode : public SynTreeNode {
private:
    std::string _symbol;
    SynTreeNode* _expr;
public:
    UnaryOperationNode(const std::string& symbol, SynTreeNode* expr);
    UnaryOperationNode(const UnaryOperationNode& uo);
    ~UnaryOperationNode();

    virtual int Interpret(SymbolTable& st) const;
    virtual void Print() const;
    virtual SynTreeNode* Clone() const;
};

class LogicalOperationNode : public SynTreeNode {
private:
    std::string _symbol;
    SynTreeNode* _left;
    SynTreeNode* _right;
public:
    LogicalOperationNode(const std::string& symbol, SynTreeNode* left, SynTreeNode* right);
    LogicalOperationNode(const LogicalOperationNode& bo);
    ~LogicalOperationNode();

    virtual int Interpret(SymbolTable& st) const;
    virtual void Print() const;
    virtual SynTreeNode* Clone() const;
};

class PrintNode : public SynTreeNode {
private:
    bool _logical;
    char _fmt;
    SynTreeNode* _expr;
public:
    PrintNode(SynTreeNode* expr, char fmt);
    PrintNode(SynTreeNode* expr, bool logical);
    PrintNode(const PrintNode& pn);
    ~PrintNode();

    virtual int Interpret(SymbolTable& st) const;
    virtual void Print() const;
    virtual SynTreeNode* Clone() const;  
};

class AssignmentNode : public SynTreeNode {
private:
    std::string _name;
    SynTreeNode* _expr;
    bool _definition;
public:
    AssignmentNode(const std::string& name, SynTreeNode* expr);
    AssignmentNode(const std::string& name, SynTreeNode* expr, bool def);
    AssignmentNode(const AssignmentNode& pn);
    ~AssignmentNode();

    virtual int Interpret(SymbolTable& st) const;
    virtual void Print() const;
    virtual SynTreeNode* Clone() const;  
};

class StatementSequenceNode : public SynTreeNode {
private:
    std::vector<SynTreeNode*> _stmts;
public:
    StatementSequenceNode() {}
    StatementSequenceNode(const StatementSequenceNode& ssn);
    ~StatementSequenceNode();

    void Push(SynTreeNode* stmt);
    virtual int Interpret(SymbolTable& st) const;
    virtual void Print() const;
    virtual SynTreeNode* Clone() const; 
};

#endif

